﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Utils;

namespace iStatVball3;

/*
 * Represents one dimension of data (player, rotation, position, etc) in a data set. A dimension can in turn have an
 * inner dimension (e.g., rotation by player). Inner dimension entries are grouped by the outer dimension key.
 */ 
public class DataDimension
{
	/* Constants */

	// Outer/inner dimension level
	public enum Level
	{
		Outer,
		Inner
	};

	/* Properties */
	public string Type { get; set; }
	public List<string> OuterKeys { get; set; }

	/* Convenience Accessors */
	public int OuterCount => OuterKeys.Count;

	/* Calculated Results */
	public readonly Dictionary<string,DataMetrics> Metrics;

	/* Methods */
	public DataDimension()
	{
		Metrics = new Dictionary<string, DataMetrics>();
		OuterKeys = [];
	}

	// Processes dimensional stats for specified stat list
	public static async Task ProcessDimension( DataConfig config, DataStats stats )
	{
		#if DEBUG
			DXProfiler.Start( true );
		#endif

		// Data set specific accumulation (default Skills)
		switch ( config.DataSet )
		{
			case DataConfig.SideoutData: MetricsSideout.AccumulateDimension( config, stats ); break;
			case DataConfig.PlayingData: MetricsPlaying.AccumulateDimension( config, stats ); break;
			case DataConfig.ScoringData: MetricsScoring.AccumulateDimension( config, stats ); break;

			default: MetricsSkillAccumulator.AccumulateDimension( config, stats ); break;
		}

		// Count sets played in normal v3 stats data
		config.Dimension.CountSets( stats );

		// Calculate total sets played
		config.Totals.CountSets( stats );

		// Must accumulate raw summary data AFTER count sets
		if ( config.IsTeam1 && (config.OuterDim == KeyDimension.PlayerKey) )
		{
			await MetricsSkillRaw.Accumulate( config );
		}

		// Calculate all statistics
		config.Dimension.Calculate();
		config.Totals.Calculate();

		#if DEBUG
			DXProfiler.Mark( "dimension", true );
		#endif
	}

	// Populates sets played for all underlying metrics
	public void CountSets( DataStats stats )
	{
		foreach ( DataMetrics metrics in Metrics.Values )
		{
			metrics.CountSets( stats );
		}
	}

	// Runs calculations for all underlying metrics
	public void Calculate()
	{
		foreach ( DataMetrics metrics in Metrics.Values )
		{
			metrics.Calculate();
		}
	}
}

//
