﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Provides segment bar interface for team selection on the dashboard form.
 */
public class DashboardTeam : DXBorder
{
	/* Constants */
	public const double DefaultSegmentWd = 175;
	
	// Segment keys
	public const string Team1Key = "team1";
	public const string Team2Key = "team2";

	/* Events */
	public Action<bool> TeamChanged { get; set; }

	/* Properties */
	public DashboardConfig Config { get; set; }
	public bool IsTeam1 => segmentBar.SelectedKey == Team1Key;

	/* Fields */
	private readonly DXSegmentBar segmentBar;

	/* Methods */
	public DashboardTeam()
	{
		bool tablet = DXDevice.IsTablet;
		
		BackgroundColor = DXColors.Dark3;
		Padding = new Thickness( 15, 5, 15, 5 );

		// Underlying layout
		DXGridLayout layout = new()
		{
			Padding = 0,
			RowSpacing = 0,
			ColumnSpacing = 0,

			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill,
			
			IsClippedToBounds = true
		};

		// Team 1/2
		segmentBar = new DXSegmentBar
		{
			Mode = DXSegmentBar.SegmentMode.Text,

			FillColor = DXColors.Dark3,
			TextColor = DXColors.Light4,
			SelectColor = DXColors.Light4,

			SegmentWd = DefaultSegmentWd,
			SegmentHt = 40,
			SegmentFontSize = 15,

			Padding = new Thickness( (tablet ? 0 : 15), 0, 0, 0 ),
			WidthRequest = ((DefaultSegmentWd * 2) + Padding.HorizontalThickness),

			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Center,

			Selected = OnSelected
		};

		segmentBar.Init();

		// Text dynamically set later
		segmentBar.AddSegmentText( Team1Key );
		segmentBar.AddSegmentText( Team2Key );

		layout.Add( segmentBar, 0, 0 );

		// Always start with Team 1
		segmentBar.SelectKey( Team1Key );
		
		Content = layout;
	}

	// Post construction initialization
	public void Init( DashboardConfig config )
	{
		string team1 = (config.Team1 == null) ? config.Organization.Name : config.Team1.ShortName;
		string team2;

		// Anonymous opponent(s)
		if ( config.Team2 == null )
		{
			string scope = config.Scope;

 			team2 = (scope is "set" or "match") ? DXString.Get( "analyze.opp" ) : DXString.Get( "analyze.opps" );
		}
		// Named opponent
		else
		{
			team2 = config.Team2.ShortName;
		}

		// Team names
		segmentBar.SetText( Team1Key, team1.ToUpper() );
		segmentBar.SetText( Team2Key, team2?.ToUpper() );

		bool disabled = Shell.CurrentUser.IsMedia && !Config.IsSample;

		// Opponent not valid for media
		segmentBar.Disable( Team2Key, disabled );

		// Always default to Team 1
		InternalSelect( true );
	}

	// Programatically selects specified team without listener callback
	private void InternalSelect( bool team1 )
	{
		string key = (team1 ? Team1Key : Team2Key);
		
		segmentBar.Selected -= OnSelected;
		
		// Select without callback
		segmentBar.SelectKey( key );
		
		segmentBar.Selected += OnSelected;
	}

	// Disables specified team
	public void Disable( bool team1, bool disabled )
	{
		segmentBar.Disable( (team1 ? Team1Key : Team2Key), disabled );
	}

	/* Event Callbacks */

	// User tapped segment
	private void OnSelected( string key )
	{
		DXSpinner.Start();
		
		bool isTeam1 = (key == Team1Key);

		// Callback registered listener
		TeamChanged?.Invoke( isTeam1 );
	}
}

//
