﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;

namespace iStatVball3;

/*
 * Displays list of up to 3 metrics on a dashboard card. Each metric displays its name and current calculated value. 
 */ 
public class DashboardMetrics : DXGridLayout
{
	/* Constants */
	public const int MaxCount = 3;
	private const string MetricsNA = "--";

	/* Properties */
	public bool IsTeam1 { get; set; }

	/* Fields */
	private readonly List<JsonMetric> jsonMetrics;

	private readonly List<DXLabel> nameLabels;
	private readonly List<DXLabel> valueLabels;

	private readonly int digits;

	/* Methods */
	public DashboardMetrics()
	{
		// Decimal precision customizable
		digits = Shell.Settings.AnalyzeDigits;

		Padding = new Thickness( 10, 10, 15, 5 );

		RowSpacing = 10;
		ColumnSpacing = 0;

		// Config
		jsonMetrics = new List<JsonMetric>( MaxCount );

		// UI
		nameLabels = new List<DXLabel>( MaxCount );
		valueLabels = new List<DXLabel>( MaxCount );

		/* Layout */

		// 2 columns
		AddStarColumn( 65 );	// 0: name
		AddStarColumn( 35 );	// 1: value

		// 3 rows
		AddStarRows( 3 );
	}

	// Adds metric to this display list
	public void AddMetric( JsonMetric metric )
	{
		jsonMetrics.Add( metric );

		int row = nameLabels.Count;

		// Name
		DXLabel nameLbl = new()
		{
			Margin = 0,

			Text = metric.Title,
			TextColor = DXColors.Dark3,

			Font = DXFonts.Roboto,
			FontSize = 14,

			LineBreakMode = LineBreakMode.TailTruncation,
			VerticalTextAlignment = TextAlignment.Center,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center
		};

		nameLabels.Add( nameLbl );

		Add( nameLbl, 0, row );

		// Value
		DXLabel valueLbl = new()
		{
			TextColor = DXColors.Dark1,

			Font = DXFonts.Roboto,
			FontSize = 14,

			LineBreakMode = LineBreakMode.TailTruncation,
			VerticalTextAlignment = TextAlignment.Center,

			Horizontal = LayoutOptions.End,
			Vertical = LayoutOptions.Center
		};

		valueLabels.Add( valueLbl );

		Add( valueLbl, 1, row );
	}

	// Updates data values for each metric in list
	public void UpdateData()
	{
		// Display formatted value for each metric
		for ( int i = 0; i < jsonMetrics.Count; i++ )
		{
			valueLabels[i].Text = GetMetricText( jsonMetrics[i] );
		}
	}

	// Returns value for key from specified team totals as formatted text
	private string GetMetricText( JsonMetric json )
	{
		// Metric data cached for all data sets
		DataMetrics totals = DashboardCache.Instance.GetMetrics( json.Data );

		// Dynamically retrieve value matching metric key
		object value = totals?.GetType().GetProperty( json.Key )?.GetValue( totals );

		if ( value != null )
		{
			// All invalid values return "--"
			switch ( json.Type )
			{
				// 1
				case AnalyzeKeys.IntKey:
				{
					int intVal = (int)value;

					return (intVal < 0) ? MetricsNA : intVal.ToString();
				}
				// 1 - 1.000
				case AnalyzeKeys.FloatKey:
				{
					float floatVal = (float)value;

					return float.IsNaN( floatVal ) ? MetricsNA : floatVal.ToString( $"N{digits}" );
				}
				// 0.100
				case AnalyzeKeys.FixedKey:
				{
					float fixedVal = (float)value;

					return float.IsNaN( fixedVal ) ? MetricsNA : fixedVal.ToString( "0.000" );
				}
				// 1% - 1.000%
				case AnalyzeKeys.PercentKey:
				{
					float percentVal = (float)value;

					return float.IsNaN( percentVal ) ? MetricsNA : percentVal.ToString( $"P{digits}" );
				}
				// +1
				case AnalyzeKeys.DeltaKey:
				{
					float floatVal = (float)value;

					return float.IsNaN( floatVal ) ? MetricsNA : floatVal.ToString( "+#;-#;0" );
				}
				// 1m 1s
				case AnalyzeKeys.TimeKey:
				{
					return (string)value;
				}
			}
		}

		return null;
	}
}

//
