﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Card;

using DXLib.UI.Control.Image;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Displays one card on the dashboard form. Each card has a category title, group icon, and list of metrics. Custom
 * dashboard cards also have an edit button.
 */
public class DashboardCard : DXCard
{
	/* Constants */
	public const double CardHt = 195;

	/* Properties */
	public string Icon { set => groupImage.SetIcon( value ); }

	public string Category { set => titleLabels.Title = value; }
	public string Group { set => titleLabels.SubTitle = value; } 

	// Custom dashboard
	public CustomConfig CustomConfig { get; set; }

	/* Field */
	private readonly DXImageArea groupImage;
	private readonly DXCardLabels titleLabels;
	private readonly DXButton editBtn;

	// 0-3 metrics displayed on card
	private readonly DashboardMetrics metrics;

	// Associated data model for custom dashboards
	private readonly CustomDashboard custom;

	/* Methods */
	public DashboardCard( CustomDashboard custom )
	{
		this.custom = custom;

		Padding = 0;
		Margin = 0;
		
		// Icon
		groupImage = new DXImageArea();

		// Category/Group
		titleLabels = new DXCardLabels
		{
			TitleLbl =
			{
				FontSize = 18
			},
			SubTitleLbl =
			{
				Font = DXFonts.Roboto,
				FontSize = 12
			}
		};

		// Metrics
		metrics = new DashboardMetrics();

		// 3 rows
		layout.AddFixedRow( 60 );       // 0: labels
		layout.AddStarRow();			// 1: metrics
		layout.AddFixedRow( 50 );       // 2: edit

		// 2 columns
		layout.AddStarColumn( 62 );     // 0: titles/metrics
		layout.AddStarColumn( 38 );     // 1: icon

		// Add components
		layout.Add( titleLabels, 0, 0 );
		layout.Add( metrics, 0, 1 );
		layout.Add( groupImage, 1, 0, 1, 2 );

		// Custom dashboard adds Edit button
		if ( (custom != null) && CustomDashboard.CanEdit( Shell.CurrentUser ) )
		{
			editBtn = new DXButton
			{
				Resource = "card.edit",
				Type = DXButton.ButtonType.Neutral,

				Margin = new Thickness( 10, 0, 10, 8 ),

				Horizontal = LayoutOptions.Start,
				Vertical = LayoutOptions.End,

				IsSticky = true,
				ButtonTapped = OnEditTapped
			};

			editBtn.Init();
			
			layout.Add( editBtn, 0, 2, 2, 1 );
		}
	}

	// Post construction initialization
	public override void Init()
	{
		base.Init();
		
		groupImage.Color = CustomConfig.Color;
	}

	// Adds new metric name/value to this card
	public void AddMetric( JsonMetric metric )
	{
		metrics.AddMetric( metric );
	}

	// Updates all data values on this card
	public void UpdateData()
	{
		metrics.UpdateData();
	}

	// Required implementation
	public override double GetViewHt()
	{
		return CardHt;
	}

	// Resets button tap state
	public override void Reset()
	{
		base.Reset();

		editBtn?.Reset();
	}

	/* Event Callbacks */

	// User editing custom dashboard
	private void OnEditTapped( object sender )
	{
		// Create edit form
		CustomForm form = new( custom )
		{
			Config = CustomConfig
		};
		
		form.Init();
		
		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}

	/* Layout */

	// Card specific draw behavior
	public override void UpdateLayout( LayoutType type )
	{
		Size size = CardSmall.GetImageSize();

		// Re-scale icon based on new layout
		groupImage.SetSize( size.Width, size.Height, 0.70 );

		// Required for full bleed
		groupImage.Margin = new Thickness( 0, -2, 0, 0 );
	}
}

//
