﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Alert;
using DXLib.UI.Control;

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Data entry form for creating/editing custom dashboards. Allows user to edit required fields. Also displays Report Builder
 * for configuring and ordering dashboard reports.
 */
public class CustomForm : DXForm
{
	/* Properties */
	public CustomConfig Config { get; set; }

	/* Fields */
	private readonly CustomControl control;
	private readonly DXFill fill;

	/* Methods */
	public CustomForm( CustomDashboard dashboard ) : base( dashboard, "dashboard" )
	{
		IsClippedToBounds = true;
		
		// Cannot edit image
		IsIconOnly = true;

		header.Title = HasData ? dashboard.Name : CreateHeader();

		/* Required */

		// Name
		AddControl( new DXTextField
		{
			Key = "name",
			Title = "form.name",
			Text = dashboard?.Name,
			MinLength = 2,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "custom.name"
		}, true, true );

		// Description
		AddControl( new DXEditorField
		{
			Key = "desc",
			Title = "form.desc",
			Text = dashboard?.Description,
			MaxLength = 1024,
			Help = "custom.desc"
		}, true, false );
		
		// Create and populate Report Builder
		control = new CustomControl( this );
		control.SetReports( dashboard?.Reports );
		
		// Background fill
		fill = new DXFill
		{
			Color = DXColors.Accent1,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};
	}

	// Post construction initialization
	public override void Init()
	{
		// Control initialization
		base.Init();

		imagePanel.Color = Config.Color;

		// Init children
		control.Config = Config;
		control.Init();

		control.IsDisabled = IsReadOnly;
	}

	// Hides data form, refreshes dashboard
	private async Task Hide( bool refresh = false )
	{
		Shell shell = Shell.Instance;
		DashboardForm form = shell.DashboardForm;

		shell.HideForm( false );

		// Refresh custom card metrics
		if ( refresh )
		{
			DashboardCache.Instance.ReadCustom();

			form.UpdateActiveLayout( Config.IsTeam1 );
		}

		await form.Show( false );
	}

	// Determines if all form data (including control) is valid
	public override bool IsValid()
	{
		return base.IsValid() && control.IsValid();
	}

	/* Event Callbacks */

	// User confirmed delete
	protected override async void OnDeleteConfirmed()
	{
		base.OnDeleteConfirmed();

		if ( data is CustomDashboard dashboard )
		{
			// Cascading delete
			await dashboard.Delete();

			// Refresh UI
			await Hide();
		}
	}

	// User confirmed cancel
	protected override async void OnCancelConfirmed()
	{
		base.OnCancelConfirmed();

		// Refresh UI
		await Hide();
	}

	// User persisting report create/edit
	protected override async void OnSaveTapped()
	{
		User user = Shell.CurrentUser;

		// Retrieve fields
		string name = GetString( "name", true );
		string desc = GetString( "desc", true );

		CustomDashboard dashboard = HasData ? (data as CustomDashboard) : null;
		
		// Custom report name MUST be unique for user
		if ( CustomDashboard.IsUnique( name, dashboard ) )
		{
			List<CustomReport> reports = control.GetReports();

			base.OnSaveTapped();

			// Update existing object
			if ( HasData && (dashboard != null) )
			{
				dashboard.Name = name;
				dashboard.Description = desc;
				dashboard.Reports = reports;

				// Persist
				await user.UpdateDashboards();
			}
			// Create new object
			else
			{
				CustomDashboard newDashboard = new()
				{
					IsTeam1 = Config.IsTeam1,

					Name = name,
					Description = desc,
					Reports = reports
				};

				// Add to parent
				user.Dashboards.Add( newDashboard );

				// Persist
				await user.UpdateDashboards();

				// Cache
				DashboardCache.Instance.AddCustom( newDashboard );
			}

			// Refresh UI
			await Hide( true );
		}
		else
		{
			DXAlert.ShowError( "dashboard.singular", "custom.err.unique", name );
		}
	}

	/* Layout */

	// Extends parent layout to add custom control
	public override void UpdateLayout( LayoutType type )
	{
		base.UpdateLayout( type );

		// Update children
		control.UpdateLayout( type );
	}

	// Landscape (4:3)
	protected override void Landscape()
	{
		base.Landscape();

		// Replace secondary with Report Builder
		Layout.Remove( SecondaryList.View );
		Layout.Remove( fill );
		
		Layout.Add( fill, 2, 1 );
		Layout.Add( control, 2, 1 );
	}

	// Portrait (3:4)
	protected override void Portrait()
	{
		base.Portrait();

		// Replace secondary with Report Builder
		Layout.Remove( SecondaryList.View );
		Layout.Remove( fill );
		
		Layout.Add( fill, 1, 2 );
		Layout.Add( control, 1, 2 );
	}

	// Mobile landscape
	protected override void MobileLandscape()
	{
		base.MobileLandscape();

		// Replace secondary with Report Builder
		LayoutL.Remove( SecondaryList.View );
		Layout.Remove( fill );
		
		Layout.Add( fill, 2, 1 );
		LayoutL.Add( control, 2, 1 );
	}

	// Mobile portrait
	protected override void MobilePortrait()
	{
		base.MobilePortrait();

		// Replace secondary with Report Builder
		LayoutP.Remove( SecondaryList.View );
		LayoutP.Add( control );
	}
}

//
