﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;

namespace iStatVball3;

/*
 * Implements the ReportBuilder control, which allows the user to create/edit multiple custom reports and add them to
 * an ordered list.
 */
public class CustomControl : DXGridLayout
{
	/* Properties */
	public CustomConfig Config { get; set; }

	// Visually disables all components
	public bool IsDisabled { set => SetDisabled( value ); }

	/* Fields */
	private readonly DXLabel addLbl;
	private readonly DXIconButton addBtn;

	// Underlying list control
	private readonly CustomList list;

	// Underlying data
	private readonly List<CustomReport> reports;

	// External refs
	private readonly CustomForm form;

	/* Methods */
	public CustomControl( CustomForm form )
	{
		this.form = form;

		// Allocate data container
		reports = [];
		
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Start;
		
		// Vertical layout
		BackgroundColor = DXColors.Accent1;

		Padding = 24;
		RowSpacing = 10;
		ColumnSpacing = 10;

		// Title
		DXLabel titleLbl = new()
		{
			Resource = "dashboard.custom",
			
			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = 17,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center
		};

		// list control
		list = new CustomList( this )
		{
			IsVisible = false,
			Margin = 0, 
			
			Completed = OnFormCompleted,
			Cancelled = OnFormCancelled,

			Reordered = OnListReordered,
			Deleted = OnItemDeleted
		};

		// 'Add new report'
		addLbl = new DXLabel
		{ 
			Resource = "dashboard.add",

			TextColor = DXColors.Dark4,
			Font = DXFonts.Roboto,
			FontSize = 14,

			Horizontal = LayoutOptions.End,
			Vertical = LayoutOptions.Center
		};

		const double addSize = 36;
		
		// '+'
		addBtn = new DXIconButton
		{
			Resource = "add",
			Color = DXColors.Positive,

			IconColor = DXColors.Light4,
			IconScale = 0.75f,

			Horizontal = LayoutOptions.End,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			HasShadow = true,

			ButtonTapped = OnAddTapped
		};
		
		addBtn.Init();
		addBtn.SetSize( addSize, true );

		// 3 rows
		AddFixedRow( 20 );				// 0: title
		AddStarRow();					// 1: list
		AddFixedRow( addSize );			// 2: add
		
		// 2 columns
		AddStarColumn();				// 0: 'Add new report'
		AddFixedColumn( addSize );		// 1: + button

		// Add components
		Add( titleLbl, 0, 0, 2, 1 );
		Add( list, 0, 1, 2, 1 );
		
		Add( addLbl, 0, 2 );
		Add( addBtn, 1, 2 );
	}

	// Post construction initialization
	public void Init()
	{
		list.Config = Config;
	}

	// Determines if control data currently valid (at least 1 report)
	public bool IsValid()
	{
		return (reports.Count > 0);
	}

	// Returns count of reports highlighted on dashboard card
	public int GetCardCount()
	{
		return reports.Count( report => report.IsOnCard );
	}

	// Visually enables/disables control components
	private void SetDisabled( bool disabled )
	{
		list.IsDisabled = disabled;

		addLbl.IsDisabled = disabled;
		addBtn.IsDisabled = disabled;
	}

	/* Reports */

	// Returns report list in current ordering
	public List<CustomReport> GetReports()
	{
		return list.GetItems();
	}

	// Sets custom report list to be displayed in control
	public void SetReports( IList<CustomReport> items )
	{
		reports.Clear();

		if ( items != null )
		{
			// Add to underlying data
			reports.AddRange( items );

			// Update UI
			list.SetItems( items );
			list.IsVisible = (items.Count > 0);
		}
	}

	/* Event Callbacks */

	// User tapped '+' button
	private void OnAddTapped()
	{
		// Create form
		CustomListForm listForm = new( this, null )
		{
			Config = Config,

			Completed = OnFormCompleted,
			Cancelled = OnFormCancelled
		};

		listForm.Init();

		// Show in modal popup
		listForm.Show();
	}

	// User finished creating report
	private void OnFormCompleted( CustomReport report, bool edit )
	{
		// Editing
		if ( edit )
		{
			int index = reports.IndexOf( report );

			// Replace existing report
			reports.RemoveAt( index );
			reports.Insert( index, report );
		}
		// Creating
		else
		{
			reports.Add( report );
		}

		// Display in list
		list.SetItems( reports );
		list.IsVisible = true;

		// Update UI
		list.Selected = null;

		addBtn.Reset();
		form.UpdateSave();
	}

	// User cancelled report creation
	private void OnFormCancelled()
	{
		list.Selected = null;

		addBtn.Reset();
	}

	// User completed drag 'n drop list reordering
	private void OnListReordered()
	{
		form.UpdateSave();
	}

	// User completed swipe to delete
	private void OnItemDeleted( CustomReport report )
	{
		// Remove from underlying data
		reports.Remove( report );

		// Update UI
		list.SetItems( reports );
		form.UpdateSave();
	}
}

//
