import 'package:flutter/material.dart';
import 'package:syncfusion_flutter_calendar/calendar.dart';
import 'package:syncfusion_localizations/syncfusion_localizations.dart';
import 'package:flutter_localizations/flutter_localizations.dart';

void main() {
  runApp(const MainApp());
}

class MainApp extends StatefulWidget {
  const MainApp({super.key});

  @override
  State<MainApp> createState() => _MainAppState();
}

class _MainAppState extends State<MainApp> {
  bool searching = false;
  List<Slot> appointments = <Slot>[];

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      localizationsDelegates: const [
        GlobalMaterialLocalizations.delegate,
        GlobalWidgetsLocalizations.delegate,
        SfGlobalLocalizations.delegate
      ],
      supportedLocales: const [
        Locale('es'),
      ],
      locale: const Locale('es'),
      home: Scaffold(
        body: Column(
          children: [
            SizedBox(
              height: 100.0,
              child: IconButton(
                  onPressed: () async {
                    setState(() {
                      searching = true;
                    });
                    await _readAppointments();
                    setState(() {
                      searching = false;
                    });
                  },
                  icon: const Icon(Icons.search)),
            ),
            Expanded(
                child: Visibility(
              visible: !searching,
              replacement: const Center(
                child: CircularProgressIndicator(),
              ),
              child: SfCalendar(
                initialSelectedDate: DateTime.now(),
                view: CalendarView.month,
                dataSource: DataSource(appointments),
                firstDayOfWeek: 1,
                showNavigationArrow: true,
                showDatePickerButton: true,
                monthViewSettings: const MonthViewSettings(
                    showAgenda: true,
                    agendaViewHeight: 500.0,
                    appointmentDisplayCount: 0,
                    showTrailingAndLeadingDates: false,
                    dayFormat: 'EEEE',
                    agendaItemHeight: 100.0),
                monthCellBuilder: (context, details) =>
                    getMonthBuilder(details),
              ),
            )),
          ],
        ),
      ),
    );
  }

  Future<void> _readAppointments() async {
    await Future.delayed(const Duration(seconds: 3));

    if (appointments.isEmpty) {
      appointments.add(Slot(
        from: DateTime(2024, 11, 12, 10),
        to: DateTime(2024, 11, 12, 12),
      ));
      appointments.add(Slot(
        from: DateTime(2024, 11, 12, 12),
        to: DateTime(2024, 11, 12, 12, 15),
      ));
      appointments.add(Slot(
        from: DateTime(2024, 11, 13, 12),
        to: DateTime(2024, 11, 13, 12, 15),
      ));
      appointments.add(Slot(
        from: DateTime(2024, 11, 14, 12, 40),
        to: DateTime(2024, 11, 14, 12, 50),
      ));
      appointments.add(Slot(
        from: DateTime(2024, 11, 12, 12, 40),
        to: DateTime(2024, 11, 12, 13, 50),
      ));
    } else {
      for (Slot s in appointments) {
        s.from = s.from.add(const Duration(days: 1));
        s.to = s.to.add(const Duration(days: 1));
      }
    }
  }

  Widget getMonthBuilder(MonthCellDetails details) {
    debugPrint(
        'Fecha: ${details.date} Nº appointments: ${details.appointments.length}');
    return Container(
      margin: const EdgeInsets.all(0.0),
      padding: EdgeInsets.zero,
      alignment: Alignment.center,
      child: Text(
        details.date.day.toString(),
        style: TextStyle(color: colorDay(details), fontSize: 10.0),
      ),
    );
  }

  Color colorDay(MonthCellDetails day) {
    return day.appointments.isNotEmpty ? Colors.green : Colors.red;
  }
}

class Slot {
  Slot({
    required this.from,
    required this.to,
  });

  DateTime from;
  DateTime to;
}

class DataSource extends CalendarDataSource<Slot> {
  DataSource(List<Slot> source) {
    appointments = source;
  }

  @override
  DateTime getStartTime(int index) {
    return appointments![index].from as DateTime;
  }

  @override
  DateTime getEndTime(int index) {
    return appointments![index].to as DateTime;
  }
}
