import 'dart:developer';

import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:font_awesome_flutter/font_awesome_flutter.dart';
import 'package:ksr_medical_manager/src/components/custom_pop_menu_button.dart';
import 'package:ksr_medical_manager/src/core/constants/colors.dart';
import 'package:ksr_medical_manager/src/core/localization/generated/l10n.dart';
import 'package:ksr_medical_manager/src/core/utils/tools.dart';
import 'package:ksr_medical_manager/src/features/skeleton/bloc/bloc/main_view_bloc.dart';
import 'package:ksr_medical_manager/src/core/models/table_column_title.dart';
import 'package:syncfusion_flutter_core/theme.dart';
import 'package:syncfusion_flutter_datagrid/datagrid.dart';

class SfDataTable extends StatelessWidget {
  const SfDataTable(
      {super.key,
      required this.myKey,
      required this.tableTitle,
      this.suffixWidget,
      required this.dataSource,
      required this.totalItems,
      required this.columnsTitles,
      this.tinyWidthColumnIndex,
      this.disableFilttering,
      this.disableSorting,
      this.excludeColumnsOnExport,
      this.isTinyMode = false});
  final GlobalKey<SfDataGridState> myKey;
  final String tableTitle;
  final Widget? suffixWidget;
  final DataGridSource dataSource;
  final int totalItems;
  final List<TableColumnTitle> columnsTitles;
  final List<int>? tinyWidthColumnIndex;
  final List<int>? disableFilttering;
  final List<int>? disableSorting;
  final List<String>? excludeColumnsOnExport;
  final bool isTinyMode;

  @override
  Widget build(BuildContext context) {
    return LayoutBuilder(builder: (context, constraints) {
      // final rowPerPage =
      //     Tools.calculateDataTableRowsPerPage(constraints.maxHeight);
      const rowPerPage = 4;
      return Card(
        child: Column(
          children: [
            DataTableHeader(
              myKey: myKey,
              title: tableTitle,
              suffixWidget: suffixWidget,
              excludeColumns: excludeColumnsOnExport,
              isTinyMode: isTinyMode,
            ),
            Expanded(
                child: Padding(
              padding: const EdgeInsets.all(1.0),
              child: DataGrid(
                dataSource: dataSource,
                myKey: myKey,
                columnsTitles: columnsTitles,
                rowPerPage: rowPerPage,
                tinyWidthColumnIndex: tinyWidthColumnIndex,
                disableFilttering: disableFilttering,
                disableSorting: disableSorting,
                isTinyMode: isTinyMode,
              ),
            )),
            DataPager(
              rowPerPage: rowPerPage,
              dataSource: dataSource,
              totalItems: totalItems,
            )
          ],
        ),
      );
    });
  }
}

class DataPager extends StatelessWidget {
  const DataPager({
    super.key,
    required this.rowPerPage,
    required this.dataSource,
    required this.totalItems,
  });

  final int rowPerPage;
  final DataGridSource dataSource;
  final int totalItems;

  @override
  Widget build(BuildContext context) {
    final pageCount = (totalItems / rowPerPage).ceilToDouble();
    return SfDataPagerTheme(
        data: const SfDataPagerThemeData(),
        child: SfDataPager(
          pageCount: pageCount > 0 ? pageCount : 1,
          delegate: dataSource,
        ));
  }
}

class DataTableHeader extends StatelessWidget {
  const DataTableHeader({
    super.key,
    required this.myKey,
    required this.title,
    this.suffixWidget,
    this.excludeColumns,
    required this.isTinyMode,
  });

  final GlobalKey<SfDataGridState> myKey;
  final String title;
  final Widget? suffixWidget;
  final List<String>? excludeColumns;
  final bool isTinyMode;

  @override
  Widget build(BuildContext context) {
    return Row(
      children: [
        Padding(
          padding: EdgeInsets.all(isTinyMode ? 12 : 24),
          child: Text(
            title,
            style: Theme.of(context)
                .textTheme
                .displaySmall
                ?.copyWith(fontSize: isTinyMode ? 16 : 20),
          ),
        ),
        const Spacer(),
        suffixWidget ?? const SizedBox(),
        const SizedBox(width: 10),
        Transform.scale(
          scale: isTinyMode ? .8 : 1,
          child: MyPopMenuButton(
            items: [
              S.of(context).exporter_en_pdf,
              S.of(context).exporter_vers_excel
            ],
            isTinyMode: isTinyMode,
            onSelected: (value) async {
              bool exported = false;
              bool? result;
              if (value == 0) {
                result = await Tools.exportDataGridToPdf(
                    key: myKey,
                    fileTitle: title,
                    excludeColumns: excludeColumns);
                if (result == null) {
                  return;
                }
                if (result) {
                  exported = true;
                }
              } else if (value == 1) {
                result = await Tools.exportDataGridToExcel(
                    key: myKey,
                    fileTitle: title,
                    excludeColumns: excludeColumns);
                if (result == null) {
                  return;
                }
                if (result) {
                  exported = true;
                }
              }

              if (exported && context.mounted) {
                ScaffoldMessenger.of(context).showSnackBar(
                  SnackBar(
                    content: Text(S.of(context).succes_a_lexportation),
                    duration: const Duration(milliseconds: 1500),
                    width: 400,
                  ),
                  snackBarAnimationStyle: AnimationStyle(
                    duration: const Duration(milliseconds: 1000),
                  ),
                );
              } else if (!exported && context.mounted) {
                ScaffoldMessenger.of(context).showSnackBar(
                  SnackBar(
                    content: Text(S.of(context).erreur_a_lexportation),
                    duration: const Duration(milliseconds: 1500),
                    width: 400,
                  ),
                  snackBarAnimationStyle: AnimationStyle(
                    duration: const Duration(milliseconds: 1000),
                  ),
                );
              }
            },
          ),
        ),
        const SizedBox(width: 10),
      ],
    );
  }
}

class DataGrid extends StatefulWidget {
  final DataGridSource dataSource;

  final GlobalKey<SfDataGridState> myKey;
  final List<int>? disableFilttering;
  final List<int>? disableSorting;
  final List<TableColumnTitle> columnsTitles;
  final int rowPerPage;
  final List<int>? tinyWidthColumnIndex;
  final bool isTinyMode;
  const DataGrid(
      {super.key,
      required this.dataSource,
      required this.myKey,
      required this.columnsTitles,
      required this.rowPerPage,
      this.tinyWidthColumnIndex,
      this.disableFilttering,
      this.disableSorting,
      required this.isTinyMode});

  @override
  State<DataGrid> createState() => _DataGridState();
}

class _DataGridState extends State<DataGrid> {
  @override
  Widget build(BuildContext context) {
    return SfDataGridTheme(
      data: SfDataGridThemeData(
        headerHoverColor: (context.read<MainViewBloc>().isDarkMode
                ? DarkColors.dashboardTileTitlesColor
                : LightColors.dashboardTileTitlesColor)
            .withOpacity(.1),
        headerColor: context.read<MainViewBloc>().isDarkMode
            ? Theme.of(context).cardColor
            : MyColors.lightBlue.withOpacity(.1),
        gridLineColor: Theme.of(context).dividerColor.withOpacity(.13),
        sortIconColor: Theme.of(context).textTheme.headlineMedium?.color,
        filterIcon: Icon(
          FontAwesomeIcons.filter,
          size: 14,
          color: Theme.of(context).textTheme.headlineMedium?.color,
        ),
        gridLineStrokeWidth: 2,
        rowHoverColor: (context.read<MainViewBloc>().isDarkMode
                ? DarkColors.dashboardTileTitlesColor
                : LightColors.dashboardTileTitlesColor)
            .withOpacity(.1),
      ),
      child: SfDataGrid(
        key: widget.myKey,
        source: widget.dataSource,
        columns: getColumns(
          columnTitle: widget.columnsTitles,
          disableFilttering: widget.disableFilttering,
          disableSorting: widget.disableSorting,
        ),
        rowsPerPage: widget.rowPerPage,
        headerRowHeight: widget.isTinyMode ? 30 : 35,
        rowHeight: widget.isTinyMode ? 40 : 49,
        allowColumnsResizing: true,
        allowFiltering: true,
        allowSorting: true,
        allowTriStateSorting: true,
        columnWidthMode: ColumnWidthMode.fill,
        onColumnResizeUpdate: (details) {
          setState(() {
            widget.columnsTitles[details.columnIndex].width = details.width;
          });
          return true;
        },
        gridLinesVisibility: GridLinesVisibility.horizontal,
        columnResizeMode: ColumnResizeMode.onResize,
        navigationMode: GridNavigationMode.row,
        showHorizontalScrollbar: true,
        verticalScrollPhysics: const BouncingScrollPhysics(),
      ),
    );
  }

  List<GridColumn> getColumns(
      {required List<TableColumnTitle> columnTitle,
      List<int>? disableFilttering,
      List<int>? disableSorting}) {
    return List.generate(
      columnTitle.length,
      (index) => GridColumn(
          columnName: columnTitle[index].name,
          width: columnTitle[index].width,
          maximumWidth: widget.tinyWidthColumnIndex != null &&
                  widget.tinyWidthColumnIndex!.contains(index)
              ? 180
              : double.nan,
          minimumWidth: 120,
          label: Container(
            alignment: AlignmentDirectional.centerStart,
            padding: const EdgeInsetsDirectional.only(start: 10),
            child: Text(
              columnTitle[index].name,
              overflow: TextOverflow.ellipsis,
              style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                  color: !(context.read<MainViewBloc>().isDarkMode)
                      ? Theme.of(context).primaryColor
                      : Colors.white.withOpacity(.8),
                  fontWeight: FontWeight.w600),
            ),
          ),
          allowFiltering: disableFilttering != null
              ? !disableFilttering.contains(index)
              : true,
          allowSorting:
              disableSorting != null ? !disableSorting.contains(index) : true,
          filterIconPosition: ColumnHeaderIconPosition.end,
          filterIconPadding: const EdgeInsetsDirectional.only(end: 10),
          sortIconPosition: ColumnHeaderIconPosition.start),
    );
  }
}
