﻿using MemoryToolkit.Maui;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Logging.Abstractions;
using SfMemoryLeaks.Models;

namespace SfMemoryLeaks
{
    public static class MauiProgramExtensions
    {
        public static void AddLeakDetection(this MauiAppBuilder builder)
        {
            ILogger<MemoryLeakDetector>? logger = builder.Services.BuildServiceProvider()
                .GetService<ILogger<MemoryLeakDetector>>();

            if (logger != null)
            {
                MemoryLeakDetector.Instance.Logger = logger;
            }

            builder.UseLeakDetection(customMonitor: MemoryLeakDetector.Instance);
        }
    }

    public class MemoryLeakDetector : IGarbageCollectionMonitor
    {
        public static IGarbageCollectionMonitor Instance { get; set; } = new MemoryLeakDetector();

        public ILogger Logger { get; set; } = NullLogger.Instance;

        public Action<CollectionTarget>? OnLeaked { get; set; }

        public Action<CollectionTarget>? OnCollected { get; set; }

        public event EventHandler<List<DetectedLeakModel>>? OnReportLeaks;

        public async Task MonitorAndForceCollectionAsync(List<CollectionTarget> collectionItems)
        {
            const int maxCollections = 10;
            const int msBetweenCollections = 200;
            var currentCollection = 0;

            List<CollectionTarget> leakedItems = [];
            List<CollectionTarget> collectedItems = [];

            while (++currentCollection <= maxCollections)
            {
                GC.Collect();
                GC.WaitForPendingFinalizers();

                foreach (CollectionTarget item in collectionItems.ToArray())
                {
                    // check if the item is still alive
                    if (item.Reference.IsAlive && currentCollection < maxCollections)
                        continue;

                    collectionItems.Remove(item);

                    // if the item is not alive, it was collected by the garbage collector
                    if (!item.Reference.IsAlive)
                    {
                        OnCollectedInternal(item);
                        collectedItems.Add(item);
                    }
                    // if the item is still alive after the maximum number of collections, it is considered a leak
                    else if (currentCollection == maxCollections)
                    {
                        OnLeakedInternal(item);
                        leakedItems.Add(item);
                    }
                }

                await Task.Delay(msBetweenCollections);
            }

            foreach (var collectedItem in collectedItems.GroupBy(t => t.Name))
            {
                if (collectedItem.Key == null) continue;
                Logger.LogInformation($"Collected: {collectedItem.Key}, Count: {collectedItem.Count()}");
            }

            List<DetectedLeakModel> detectedLeaks = [];

            foreach (var leakedGroup in leakedItems.GroupBy(t => t.Reference.Target?.ToString()))
            {
                if (leakedGroup.Key == null) continue;

                string leakMessage = $"Leak detected: {leakedGroup.Key}, Count: {leakedGroup.Count()}";
                Logger.LogInformation(leakMessage);

                detectedLeaks.Add(new DetectedLeakModel { Message = leakMessage });                
            }

            Logger.LogInformation($"Current memory usage: " + GC.GetTotalMemory(false) / (1024.0 * 1024.0) + " MB");

            OnReportLeaks?.Invoke(this, detectedLeaks);
        }

        protected virtual void OnLeakedInternal(CollectionTarget target)
        {
            OnLeaked?.Invoke(target);            
        }

        private protected virtual void OnCollectedInternal(CollectionTarget target)
        {
            OnCollected?.Invoke(target);
        }
    }
}
