﻿using Syncfusion.Blazor;
using Syncfusion.Blazor.Data;

namespace Extensions
{
    public static class DataManagerRequestExtensions
    {
        public static async Task<object> ReadDataAsync<T>(
            this DataManagerRequest dm,
            IQueryable<T> data,
            CancellationToken cancellationToken)
        {
            if (dm.Search?.Count > 0)
            {
                // Searching
                data = QueryableOperation.PerformSearching(data, dm.Search);
            }

            if (dm.Sorted?.Count > 0)
            {
                data = QueryableOperation.PerformSorting(data, dm.Sorted);
            }

            if (dm.Where?.Count > 0)
            {
                // Filtering
                data = QueryableOperation.PerformFiltering(data, dm.Where, dm.Where[0].Operator);
            }

            // Count (if needed) before skip and take
            var count = dm.RequiresCounts
                ? data.Count()
                : -1;

            // Aggregate (if needed) before skip and take.
            // We don't use Syncfusion's DataUtil.PerformAggregation(...) because that will fire 3 select queries and will do
            // the sum in memory. This is obviously very slow and will use a lot of memory.
            var aggregates =
                dm.Aggregates != null &&
                // Only perform the aggregation when we don't know the count or when the count is greater than 0.
                // So we don't execute queries that will return 0 anyways.
                (!dm.RequiresCounts || count > 0)
                ? DataUtil.PerformAggregation(data, dm.Aggregates)
                : null;

            if (dm.Skip != 0)
            {
                data = QueryableOperation.PerformSkip(data, dm.Skip);
            }

            if (dm.Take != 0)
            {
                data = QueryableOperation.PerformTake(data, dm.Take);
            }

            // If we export grid to Excel and aggregates is not null we also must return the DataResult because of a check in Syncfusion.
            // In that case dm.RequiresCounts must also be true, so Syncfusion will use the DataResult.
            //if (aggregates != null)
            //{
            //    dm.RequiresCounts = true;
            //}

            return dm.RequiresCounts
                ? new DataResult
                {
                    // No need to execute the query when there aren't any records anyways:
                    Result = count == 0 ? Array.Empty<T>() : data.ToList(),
                    Count = count,
                    Aggregates = aggregates,
                }
                : data.ToList();
        }
    }
}
