﻿using System.Drawing;

namespace Shared;

public static class ColorPallet
{
	/// <summary>
	/// Human readable pallet names.
	/// </summary>
	public const string
		Blue = "#5993fd",
		Indigo = "#675aa9", // Other holiday
		Purple = "#6559cc",
		Pink = "#DE64AC",
		Red = "#f1556c",
		Orange = "#fd7e14",
		Yellow = "#f7b84b", // Public holiday
		Green = "#1abc9c", // School holiday
		Teal = "#02a8b5",
		Cyan = "#37cde6", // Weekend
		EcoDriverGreen = "#5BCB18",
		EcoDriverPink = "#DE64AC",
		EcoDriverGreyBlue = "#2B2D43",
		EcoDriverGrey = "#6C6C7B",
		White = "#fff",
		Gray100 = "#f1f5f7",
		Gray200 = "#f7f7f7",
		Gray300 = "#dee2e6",
		Gray400 = "#ced4da",
		Gray500 = "#adb5bd",
		Gray600 = "#98a6ad",
		Gray700 = "#6c757d",
		Gray800 = "#343a40",
		Gray900 = "#323a46",
		Black = "#000",
		LegacyDefaultMeterColor = "#0000FF",
		LegacyDefaultWeekendColor = "#00FFF3",
		LegacyDefaultPublicHolColor = "#FF3300",
		LegacyDefaultSchoolHolColor = "#009900",
		LegacyDefaultOtherHolColor = "#FFFF00",
		WeekendColor = Cyan,
		PublicHolidayColor = Indigo,
		SchoolHolidayColor = Green,
		OtherHolidayColor = Yellow,
		DefaultMeterColor = Blue,
		DefaultTargetColor = Red,
		DefaultCostColor = Teal,
		DefaultCO2eColor = Purple;

	public static List<string> GetChartPallet()
	{
		return [
			Blue, // Default meter color
			Red, // Avoid swapping this so it's used for the target line.
			Purple,
			Orange,
			Teal,
			Pink,
			ChangeColorBrightness(Blue, 0.2f),
			ChangeColorBrightness(Red, 0.2f),
			ChangeColorBrightness(Purple, 0.2f),
			ChangeColorBrightness(Orange, 0.2f),
			ChangeColorBrightness(Teal, 0.2f),
			ChangeColorBrightness(Pink, 0.2f),
			ChangeColorBrightness(Blue, -0.2f),
			ChangeColorBrightness(Red, -0.2f),
			ChangeColorBrightness(Purple, -0.2f),
			ChangeColorBrightness(Orange, -0.2f),
			ChangeColorBrightness(Teal, -0.2f),
			ChangeColorBrightness(Pink, -0.2f)
		];
	}

	/// <summary>
	/// Gets the theme colors.
	/// </summary>
	/// <returns>System.Collections.Generic.List&lt;string&gt;.</returns>
	public static List<string> GetThemeColors()
	{
		return
		[
			Blue,
			Indigo,
			Purple,
			Pink,
			Red,
			Orange,
			Yellow,
			Green,
			Teal,
			Cyan
		];
	}

	/// <summary>
	/// Gets the corporate colors.
	/// </summary>
	/// <returns>System.Collections.Generic.List&lt;string&gt;.</returns>
	public static List<string> GetCorporateColors()
	{
		return
		[
			EcoDriverGreen,
			EcoDriverPink,
			EcoDriverGreyBlue,
			EcoDriverGrey
		];
	}

	/// <summary>
	/// Gets the neutral colors (white, greys, black).
	/// </summary>
	/// <returns>System.Collections.Generic.List&lt;string&gt;.</returns>
	public static List<string> GetNeutralColors()
	{
		return
		[
			White,
			Gray100,
			Gray200,
			Gray300,
			Gray400,
			Gray500,
			Gray600,
			Gray700,
			Gray800,
			Gray900,
			Black
		];
	}

	/// <summary>
	/// Gets the legacy colors.
	/// </summary>
	/// <returns>System.Collections.Generic.List&lt;string&gt;.</returns>
	public static List<string> GetLegacyColors()
	{
		return
		[
			LegacyDefaultMeterColor,
			LegacyDefaultWeekendColor,
			LegacyDefaultPublicHolColor,
			LegacyDefaultSchoolHolColor,
			LegacyDefaultOtherHolColor
		];
	}

	/// <summary>
	/// Changes the color brightness.
	/// </summary>
	/// <param name="hexColor">Color of the hexadecimal.</param>
	/// <param name="correctionFactor">The correction factor, must be between -1 and 1. Negative is darker, Positive is lighter.</param>
	/// <returns>Color hex code.</returns>
	/// <exception cref="ArgumentOutOfRangeException">nameof(correctionFactor), correctionFactor must be between -1 and 1</exception>
	public static string ChangeColorBrightness(string hexColor, float correctionFactor)
	{
		Color color = ColorTranslator.FromHtml(hexColor);
		var newColor = ChangeColorBrightness(color, correctionFactor);

		return ColorTranslator.ToHtml(newColor);
	}

	/// <summary>
	/// Changes the color brightness.
	/// </summary>
	/// <param name="color">The color.</param>
	/// <param name="correctionFactor">The correction factor, must be between -1 and 1. Negative is darker, Positive is lighter.</param>
	/// <returns>System.Drawing.Color.</returns>
	/// <exception cref="ArgumentOutOfRangeException">nameof(correctionFactor), correctionFactor must be between -1 and 1</exception>
	public static Color ChangeColorBrightness(Color color, float correctionFactor)
	{
		if (correctionFactor < -1 || correctionFactor > 1)
		{
			throw new ArgumentOutOfRangeException(nameof(correctionFactor), "correctionFactor must be between -1 and 1");
		}

		float red = (float)color.R;
		float green = (float)color.G;
		float blue = (float)color.B;

		if (correctionFactor < 0)
		{
			correctionFactor = 1 + correctionFactor;
			red *= correctionFactor;
			green *= correctionFactor;
			blue *= correctionFactor;
		}
		else
		{
			red = (255 - red) * correctionFactor + red;
			green = (255 - green) * correctionFactor + green;
			blue = (255 - blue) * correctionFactor + blue;
		}

		return Color.FromArgb(color.A, (int)red, (int)green, (int)blue);
	}
}


