﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.OData.Deltas;
using Microsoft.AspNetCore.OData.Formatter;
using Microsoft.AspNetCore.OData.Query;
using Microsoft.AspNetCore.OData.Results;
using Microsoft.AspNetCore.OData.Routing.Controllers;
using Microsoft.EntityFrameworkCore;
using SyncfusionExamples.Shared;
using System;
using System.Linq;
using System.Threading.Tasks;

namespace SyncfusionExamples.Server.Controllers
{
    public class OrderStatusesController : ODataController
    {
        protected readonly AppDbContext _dbContext;
        protected readonly DbSet<OrderStatus> _dbSet;

        public OrderStatusesController(AppDbContext context)
        {
            _dbContext = context;
            _dbSet = _dbContext.Set<OrderStatus>();

            var testOrder = _dbSet.Find((long)1);
            if (testOrder == null)
            {
                _dbSet.Add(new OrderStatus { Status = "Submitted"});
                _dbSet.Add(new OrderStatus { Status = "Received" });
                _dbSet.Add(new OrderStatus { Status = "Warehouse" });
                _dbSet.Add(new OrderStatus { Status = "Shipped" });
                _dbSet.Add(new OrderStatus { Status = "Delivered" });
            }

            _dbContext.SaveChanges();
        }

        async Task<bool> RecordExists(long key)
        {
            return await _dbSet.AnyAsync(p => p.OrderStatusId == key);
        }

        [HttpGet]
        [EnableQuery(AllowedQueryOptions = AllowedQueryOptions.All, MaxExpansionDepth = 0)]
        public IQueryable<OrderStatus> Get()
        {
            var all = _dbSet.ToList();
            return _dbSet;
        }


        [EnableQuery]
        public SingleResult<OrderStatus> Get([FromODataUri] long key)
        {
            IQueryable<OrderStatus> result = _dbSet.Where(p => p.OrderStatusId == key);
            return SingleResult.Create(result);
        }

        public async Task<IActionResult> Post([FromBody] OrderStatus item)
        {
            if (!ModelState.IsValid)
            {
                return BadRequest(ModelState);
            }

            _dbSet.Add(item);
            await _dbContext.SaveChangesAsync();
            return new CreatedODataResult<OrderStatus>(item);
        }

        public async Task<IActionResult> Patch([FromODataUri] long key, Delta<OrderStatus> item)
        {
            if (!ModelState.IsValid)
            {
                return BadRequest(ModelState);
            }
            var entity = await _dbSet.FindAsync(key);
            if (entity == null)
            {
                return NotFound();
            }
            item.Patch(entity);
            try
            {
                await _dbContext.SaveChangesAsync();
            }
            catch (DbUpdateConcurrencyException)
            {
                var exists = await RecordExists(key);
                if (!exists)
                {
                    return NotFound();
                }
                else
                {
                    throw;
                }
            }
            return new UpdatedODataResult<OrderStatus>(entity);
        }
        public async Task<IActionResult> Put([FromODataUri] long key, OrderStatus item)
        {
            if (!ModelState.IsValid)
            {
                return BadRequest(ModelState);
            }
            if (key != item.OrderStatusId)
            {
                return BadRequest();
            }
            _dbContext.Entry(item).State = EntityState.Modified;
            try
            {
                await _dbContext.SaveChangesAsync();
            }
            catch (DbUpdateConcurrencyException)
            {
                var exists = await RecordExists(key);
                if (!exists)
                {
                    return NotFound();
                }
                else
                {
                    throw;
                }
            }
            return new UpdatedODataResult<OrderStatus>(item);
        }

        public async Task<IActionResult> Delete([FromODataUri] long key)
        {
            var item = _dbSet.FindAsync(key);

            if (item.Result != null)
                _dbSet.Remove(item.Result);
            else
                return NotFound();

            await _dbContext.SaveChangesAsync();

            return NoContent();
        }
    }
}