﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Runtime.CompilerServices;
using System.Windows.Input;

namespace ThemedMenuTest
{
    public class Bindable<T> : ViewModelBase
    {

        private T _value;
        public Bindable(T val) { _value = val; }

        public Action<Bindable<T>>? OnSet { get; set; } = null;

        public T Value
        {
            get => _value;
            set
            {
                SetProperty(ref _value, value);
                OnSet?.Invoke(this);
            }
        }

        // Important that this be explicit, else it would be trivially easy for calling
        // code to accidentally overwrite the reference without a peep from the compiler.
        public static explicit operator Bindable<T>(T val) => new(val);
        public static implicit operator T(Bindable<T> val) => val.Value;
    }
    public class ViewModelBase : INotifyPropertyChanged
    {
        public event PropertyChangedEventHandler? PropertyChanged;

        protected virtual bool SetProperty<T>(ref T storage, T value, [CallerMemberName] string propertyName = "")
        {
            if (EqualityComparer<T>.Default.Equals(storage, value))
            {
                return false;
            }
            storage = value;
            OnPropertyChanged(propertyName);
            return true;
        }

        protected virtual void OnPropertyChanged([CallerMemberName] string propertyName = null)
        {
            //Debug.WriteLine($"OnPropertyChanged: {propertyName ?? string.Empty}");
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
    }
    public class MenuItemViewModel : ViewModelBase, ICommand
    {
        public MenuItemViewModel() { }
        public MenuItemViewModel(string title) : this(title, "") { }

        public MenuItemViewModel(string title, string inputGesture)
        {
            Header.Value = title;
            InputGesture.Value = inputGesture;
            Command.Value = this;
        }

        public MenuItemViewModel(string title, string inputGesture, ICommand command)
        {
            Header.Value = title;
            InputGesture.Value = inputGesture;
            Command.Value = command;
        }

        public Bindable<string> Header { get; set; } = (Bindable<string>)string.Empty;
        public Bindable<string> InputGesture { get; set; } = (Bindable<string>)string.Empty;
        public Bindable<bool> IsEnabled { get; set; } = (Bindable<bool>)true;
        public Bindable<ICommand> Command { get; set; } = new Bindable<ICommand>(null);

        public object CommandParameter { get; set; }

        public ObservableCollection<MenuItemViewModel> MenuItems { get; set; } = new ObservableCollection<MenuItemViewModel>();

        public event EventHandler? CanExecuteChanged;

        public void AddSeparator() => MenuItems.Add(new MenuItemViewModel("Separator"));

        public Predicate<object> CanExecutePredicate { get; set; }
        public Action<object> ExecuteAction { get; set; }

        public bool CanExecute(object? parameter) => CanExecutePredicate?.Invoke(CommandParameter) ?? true;
        public void Execute(object? parameter) => ExecuteAction?.Invoke(CommandParameter);
    }
}
