﻿using Syncfusion.DocIO.DLS;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace MergeDocs
{
    class Program
    {
        static void Main(string[] args)
        {
            using (WordDocument destinationDoc = new WordDocument("SampleDestination.docx"))
            {
                using (WordDocument sourceDoc = new WordDocument("SampleSource.docx"))
                {
                    ResolveBookmark(destinationDoc, sourceDoc, "TEST");
                    CleanupStyles(destinationDoc);
                    sourceDoc.Close();
                }

                destinationDoc.Save("SampleResult.docx");
                destinationDoc.Close();
            }

            Process.Start("SampleResult.docx");
        }


        /// <summary>
        /// Resolves the bookmark.
        /// </summary>
        /// <param name="destinationDoc">The destination document.</param>
        /// <param name="sourceDoc">The source document.</param>
        /// <param name="bookmarkName">Name of the bookmark.</param>
        public static void ResolveBookmark(WordDocument destinationDoc, WordDocument sourceDoc, string bookmarkName)
        {
            Bookmark bookmark = destinationDoc.Bookmarks.FindByName(bookmarkName);
            if (bookmark == null)
            {
                return;
            }

            // Collect the whole content from the template
            TextBodyPart textBodyPart = new TextBodyPart(destinationDoc);
            foreach (Entity wordTemplateChildEntity in sourceDoc.ChildEntities)
            {
                WSection section = wordTemplateChildEntity as WSection;
                if (section != null)
                {
                    foreach (Entity sectionChildEntity in section.ChildEntities)
                    {
                        WTextBody body = sectionChildEntity as WTextBody;
                        if (body != null)
                        {
                            // Ignore header and footer
                            if (!(body is HeaderFooter))
                            {
                                foreach (Entity bodyChildEntity in body.ChildEntities)
                                {
                                    textBodyPart.BodyItems.Add(bodyChildEntity.Clone());
                                }
                            }
                        }
                        else
                        {
                            textBodyPart.BodyItems.Add(sectionChildEntity.Clone());
                        }
                    }
                }
            }
            
            // Replace content of the bookmark with the content loaded from the template
            BookmarksNavigator bookmarkNavigator = new BookmarksNavigator(destinationDoc);
            bookmarkNavigator.MoveToBookmark(bookmark.Name);
            bookmarkNavigator.ReplaceBookmarkContent(textBodyPart);

            // Bookmark isn't needed anymore so delete it
            destinationDoc.Bookmarks.Remove(bookmark);
        }


        /// <summary>
        /// Cleanups the styles.
        /// </summary>
        /// <param name="wordDocument">The word document.</param>
        public static void CleanupStyles(WordDocument wordDocument)
        {
            // Assign the styles of the destination document if style names are the same
            List<string> stylesToDelete = new List<string>();
            foreach (WSection section in wordDocument.Sections)
            {
                IterateAllSectionEntities(section, true, true, (entity) =>
                {
                    if (entity is WParagraph paragraph && 
                        !string.IsNullOrEmpty(paragraph.StyleName))
                    {
                        string originalStyleName = paragraph.StyleName;
                        Match match = Regex.Match(originalStyleName, "(?<StyleName>.+)_[a-z0-9]{8}-[a-z0-9]{4}-[a-z0-9]{4}-[a-z0-9]{4}-[a-z0-9]{12}$", RegexOptions.IgnoreCase | RegexOptions.Compiled);
                        if (match.Success)
                        {
                            IStyle style = wordDocument.Styles.FindByName(match.Groups["StyleName"].Value);
                            if (style != null)
                            {
                                paragraph.ApplyStyle(style.Name);
                                stylesToDelete.Add(originalStyleName);
                            }
                        }
                    }

                    return false;
                });
            }

            // Delete uniquely named styles which are substituted by the styles of the destination document
            foreach (string styleToDelete in stylesToDelete)
            {
                IStyle style = wordDocument.Styles.FindByName(styleToDelete);
                if (style != null)
                {
                    // If I save and close the document before calling CleanupStyles no exception is thrown here,
                    // but the style still exists after calling style.Remove()
                    style.Remove();
                }
            }
        }


        /// <summary>
        /// Iterates all section entities.
        /// </summary>
        /// <param name="section">The section.</param>
        /// <param name="callback">The callback.</param>
        /// <param name="includeHeader">if set to <c>true</c> header is included.</param>
        /// <param name="includeFooter">if set to <c>true</c> footer is included.</param>
        /// <returns><c>true</c> if callback returns true which results in aborting the iteration; otherwise <c>false</c>.</returns>
        private static bool IterateAllSectionEntities(IWSection section, bool includeHeader, bool includeFooter, Func<Entity, bool> callback)
        {
            if (includeHeader)
            {
                if (IterateEntities(section.HeadersFooters.Header.ChildEntities, callback))
                {
                    return true;
                }

                if (IterateEntities(section.HeadersFooters.FirstPageHeader.ChildEntities, callback))
                {
                    return true;
                }

                if (IterateEntities(section.HeadersFooters.EvenHeader.ChildEntities, callback))
                {
                    return true;
                }

                if (IterateEntities(section.HeadersFooters.OddHeader.ChildEntities, callback))
                {
                    return true;
                }
            }

            if (IterateEntities(section.Body.ChildEntities, callback))
            {
                return true;
            }

            if (includeFooter)
            {
                if (IterateEntities(section.HeadersFooters.Footer.ChildEntities, callback))
                {
                    return true;
                }

                if (IterateEntities(section.HeadersFooters.FirstPageFooter.ChildEntities, callback))
                {
                    return true;
                }

                if (IterateEntities(section.HeadersFooters.EvenFooter.ChildEntities, callback))
                {
                    return true;
                }

                if (IterateEntities(section.HeadersFooters.OddFooter.ChildEntities, callback))
                {
                    return true;
                }
            }

            return false;
        }



        /// <summary>
        /// Iterates the entities.
        /// </summary>
        /// <param name="entities">The entities.</param>
        /// <param name="callback">The callback.</param>
        /// <returns><c>true</c> if callback returns true which results in aborting the iteration; otherwise <c>false</c>.</returns>
        private static bool IterateEntities(EntityCollection entities, Func<Entity, bool> callback)
        {
            // Iterates through each given child item
            for (int i = 0; i < entities.Count; i++)
            {
                // IEntity is the basic unit in DocIO DOM
                IEntity entity = entities[i];
                switch (entity.EntityType)
                {
                    // Processes the paragraph content by iterating through the paragraphs DOM
                    case EntityType.Paragraph:
                        WParagraph paragraph = entity as WParagraph;
                        if (callback(paragraph))
                        {
                            return true;
                        }

                        for (int j = 0; j < paragraph.ChildEntities.Count; j++)
                        {
                            if (callback(paragraph.ChildEntities[j]))
                            {
                                return true;
                            }

                            // Special case: Text boxes
                            if (paragraph.ChildEntities[j] is WTextBox textBox)
                            {
                                if (IterateEntities(textBox.ChildEntities, callback))
                                {
                                    return true;
                                }
                            }
                            // Special case: Inline content controls
                            else if (paragraph.ChildEntities[j] is InlineContentControl inlineContentControl)
                            {
                                if (IterateEntities(inlineContentControl.ParagraphItems, callback))
                                {
                                    return true;
                                }
                            }
                        }
                        break;

                    // Table is a collection of rows and cells so iterate through tables DOM
                    case EntityType.Table:
                        WTable table = entity as WTable;
                        if (callback(table))
                        {
                            return true;
                        }

                        foreach (WTableRow row in table.Rows)
                        {
                            foreach (WTableCell cell in row.Cells)
                            {
                                // Table cell is derived from (also a) TextBody with entities 
                                // so reusing the code meant for iterating the entities
                                if (IterateEntities(cell.ChildEntities, callback))
                                {
                                    return true;
                                }
                            }
                        }
                        break;

                    // Iterates trough the body items of a block content control
                    case EntityType.BlockContentControl:
                        BlockContentControl blockContentControl = entity as BlockContentControl;
                        if (IterateEntities(blockContentControl.TextBody.ChildEntities, callback))
                        {
                            return true;
                        }
                        break;
                }
            }

            return false;
        }
    }
}
