﻿using Syncfusion.DocIO.DLS;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace UnlinkField
{
    class Program
    {
        static void Main(string[] args)
        {

            using (WordDocument doc = new WordDocument("Sample.docx"))
            {
                foreach (WSection section in doc.Sections)
                {
                    // Start scanning for fields from beginning each time a field has been unlinked to support nested fields
                    while (IterateEntities(section.Body.ChildEntities, (entity) =>
                    {
                        if (entity is WField field)
                        {
                            field.Update();
                            field.Unlink();
                            return true;
                        }

                        return false;
                    })) ;
                }

                doc.Save("Sample_unlinked.docx");
                doc.Close();
            }

            Process.Start("Sample_unlinked.docx");
        }



        /// <summary>
        /// Iterates the entities.
        /// </summary>
        /// <param name="entities">The entities.</param>
        /// <param name="callback">The callback.</param>
        /// <returns><c>true</c> if callback returns true which results in aborting the iteration; otherwise <c>false</c>.</returns>
        private static bool IterateEntities(EntityCollection entities, Func<Entity, bool> callback)
        {
            // Iterates through each given child item
            for (int i = 0; i < entities.Count; i++)
            {
                // IEntity is the basic unit in DocIO DOM
                IEntity entity = entities[i];
                switch (entity.EntityType)
                {
                    // Processes the paragraph content by iterating through the paragraphs DOM
                    case EntityType.Paragraph:
                        WParagraph paragraph = entity as WParagraph;
                        if (callback(paragraph))
                        {
                            return true;
                        }

                        for (int j = 0; j < paragraph.ChildEntities.Count; j++)
                        {
                            if (callback(paragraph.ChildEntities[j]))
                            {
                                return true;
                            }

                            // Special case: Text boxes
                            if (paragraph.ChildEntities[j] is WTextBox textBox)
                            {
                                if (IterateEntities(textBox.ChildEntities, callback))
                                {
                                    return true;
                                }
                            }
                        }
                        break;

                    // Table is a collection of rows and cells so iterate through tables DOM
                    case EntityType.Table:
                        WTable table = entity as WTable;
                        if (callback(table))
                        {
                            return true;
                        }

                        foreach (WTableRow row in table.Rows)
                        {
                            foreach (WTableCell cell in row.Cells)
                            {
                                // Table cell is derived from (also a) TextBody with entities 
                                // so reusing the code meant for iterating the entities
                                if (IterateEntities(cell.ChildEntities, callback))
                                {
                                    return true;
                                }
                            }
                        }
                        break;

                    // Iterates trough the body items of a block content control
                    case EntityType.BlockContentControl:
                        BlockContentControl blockContentControl = entity as BlockContentControl;
                        if (IterateEntities(blockContentControl.TextBody.ChildEntities, callback))
                        {
                            return true;
                        }
                        break;
                }
            }

            return false;
        }
    }
}
