﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

using Syncfusion.Blazor;
using Syncfusion.Blazor.Data;

namespace SchedulerQueryCrash
{
    public class SimpleODataAdaptor<T> : DataAdaptor where T : ITest, new()
    {
        public IList<T> Entities { get; set; }

        public event Action OnChange;
        protected void NotifyStateChanged() => OnChange?.Invoke();

        protected async Task<IList<T>> LoadEntities()
        {
            // Here would a API request be fired
            var result = await Task.FromResult(Enumerable.Range(1, 30).Select(x => new T
            {
                Id = Guid.NewGuid(),
                TestString = $"Test: {x}"
            }).ToList());

            NotifyStateChanged();

            return result;
        }

        protected Task SimulatedApiCall(string action)
        {
            return Task.CompletedTask;
        }


        public override object Read(DataManagerRequest dm, string key = null)
        {
            return ReadAsync(dm, key).Result;
        }

        public override async Task<object> ReadAsync(DataManagerRequest dm, string key = null)
        {
            Entities ??= await LoadEntities();

            IEnumerable<T> DataSource = Entities;

            if (dm.Search != null && dm.Search.Count > 0)
            {
                // Searching
                DataSource = DataOperations.PerformSearching(DataSource, dm.Search);
            }
            if (dm.Sorted != null && dm.Sorted.Count > 0)
            {
                // Sorting
                DataSource = DataOperations.PerformSorting(DataSource, dm.Sorted);
            }
            if (dm.Where != null && dm.Where.Count > 0)
            {
                // Filtering
                DataSource = DataOperations.PerformFiltering(DataSource, dm.Where, dm.Where[0].Operator);
            }
            if (dm.Skip != 0)
            {
                //Paging
                DataSource = DataOperations.PerformSkip(DataSource, dm.Skip);
            }
            if (dm.Take != 0)
            {
                DataSource = DataOperations.PerformTake(DataSource, dm.Take);
            }

            return dm.RequiresCounts ? new DataResult() { Result = DataSource, Count = DataSource.Count() } : DataSource;
        }

        public override object Insert(DataManager dataManager, object data, string key)
        {
            return InsertAsync(dataManager, data, key).Result;
        }

        public override async Task<object> InsertAsync(DataManager dataManager, object value, string key)
        {
            var newEntity = (T)value;

            //Simulated insert API call
            await SimulatedApiCall("Insert");

            //Add new entity to entities
            Entities?.Add(newEntity);

            //Notify that the state has changed
            NotifyStateChanged();

            return newEntity;
        }

        public override object Update(DataManager dataManager, object data, string keyField, string key)
        {
            return UpdateAsync(dataManager, data, keyField, key).Result;
        }

        public override async Task<object> UpdateAsync(DataManager dataManager, object value, string keyField, string key)
        {
            var entity = (T)value;

            //Simulated remove API call
            await SimulatedApiCall("Update");

            //Replace entity in list by the edited entity
            if (Entities != null)
            {
                int index = Entities.IndexOf(Entities.FirstOrDefault(x => x.Id == entity.Id));
                if (index >= 0)
                    Entities[index] = entity;
            }

            //Notify that the state has changed
            NotifyStateChanged();

            return entity;
        }

        public override object Remove(DataManager dataManager, object data, string keyField, string key)
        {
            return RemoveAsync(dataManager, data, keyField, key).Result;
        }

        public override async Task<object> RemoveAsync(DataManager dataManager, object value, string keyField, string key)
        {
            Guid id = (Guid)value;

            //Simulated remove API call
            await SimulatedApiCall("Remove");
            
            //Remove entity from collection
            Entities?.Remove(Entities.FirstOrDefault(x => x.Id == id));

            //Notify that the state has changed
            NotifyStateChanged();

            return value;
        }

        public override object BatchUpdate(DataManager dataManager, object changedRecords, object addedRecords, object deletedRecords, string keyField, string key, int? dropIndex)
        {
            return BatchUpdateAsync(dataManager, changedRecords, addedRecords, deletedRecords, keyField, key, dropIndex).Result;
        }

        public override async Task<object> BatchUpdateAsync(DataManager dataManager, object changedRecords, object addedRecords, object deletedRecords, string keyField, string key, int? dropIndex)
        {
            await HandleRecords(addedRecords as IEnumerable<T>, (rec) => InsertAsync(dataManager, rec, key));
            await HandleRecords(changedRecords as IEnumerable<T>, (rec) => UpdateAsync(dataManager, rec, keyField, key));
            await HandleRecords(deletedRecords as IEnumerable<T>, (rec) => RemoveAsync(dataManager, rec, keyField, key));

            return new { };
        }

        private async static Task HandleRecords(IEnumerable<T> records, Func<T, Task> action)
        {
            if (records != null)
            {
                foreach (T rec in records)
                {
                    await action(rec);
                }
            }
        }
    }
}