﻿using DataAccessLibrary;
using DataAccessLibrary.DataAccess;
using DataAccessLibrary.Models;
using Microsoft.AspNetCore.Components;
using ReportsWebAppUI.Data;
using Syncfusion.Blazor;
using Syncfusion.Blazor.Grids;
using Syncfusion.Blazor.Inputs;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace ReportsWebAppUI.Pages
{
    public partial class BuildingRisk
    {
        [Inject]
        IBuildingRiskData _buildingDb { get; set; }

        [Inject]
        IPeopleData _peopleDb { get; set; }

        //variables for use on page
        private List<PersonModel> managerList;
        private List<BuildingRiskModel> buildingRiskDb;

        //data that populates the grid that is displayed to the user
        public List<EditBuildingRiskVM> buildingRiskVm = new List<EditBuildingRiskVM>();

        private SfGrid<EditBuildingRiskVM> buildingRiskGrid;
        private string selectedManager;

        //control various display dialog / notifications
        private bool showingDbError;
        private bool showingDbSuccess;

        #region Page Action - Load, toggle display elements etc
        /// <summary>
        /// When page loaded, read list of managers from Db to populate the search box
        /// </summary>
        /// <returns></returns>
        protected override async Task OnInitializedAsync()
        {

            managerList = await _peopleDb.GetPeople();
            showingDbError = false;
            showingDbSuccess = false;
        }
        #endregion ///Page Action

        #region Data Read, Loading and Manipulation
        /// <summary>
        /// Read building data, then create call 
        /// </summary>
        /// <returns></returns>
        private async Task LoadBuildingData()
        {
            //read Db to get the data
            buildingRiskDb = await _buildingDb.GetBuildingRiskData(selectedManager);
            //convert data from Db friendly model to ViewModel better suited to UI
            buildingRiskVm = LoadBuildingRiskVM(buildingRiskDb);
        }

        /// <summary>
        /// Convert the Db list of buildings to UI friendly VM that is to be displayed in the DataGrid
        /// </summary>
        /// <param name="dbBuildingRiskModels">Take in a list of type database model</param>
        /// <returns>List of BuildingRiskVM objects</returns>
        public List<EditBuildingRiskVM> LoadBuildingRiskVM(List<BuildingRiskModel> dbBuildingRiskModels)
        {
            List<EditBuildingRiskVM> riskVmList = new List<EditBuildingRiskVM>();
            if (dbBuildingRiskModels != null)
            {
               EditBuildingRiskVM riskVM = new EditBuildingRiskVM();
                    //... lots of logic to convert the data coming from the database to a ViewModel that makes sense to the user
                    // create your own version of the EditBuildingRiskVM
            
                    //finally add it to the list
                riskVmList.Add(riskVM);
                
            }
            // on error - notify the user that there is no data
            else
            {
                EditBuildingRiskVM riskVM = new EditBuildingRiskVM();
                riskVM.RecordNumber = 0;
                riskVM.BuildingNumber = "This user has no buildings assigned to them";
                riskVmList.Add(riskVM);
            }
            dbBuildingRiskModels = new List<BuildingRiskModel>();
            return riskVmList;
        }

        /// <summary>
        /// Takes changes in the data grid and applies them to the underlying model before calling dbase update method. 
        /// Called when user presses the Update button on the data grid.
        /// </summary>
        /// <param name="args">Passed in by the DataGrid</param>
        /// <returns></returns>
        public async Task BatchSaveHandlerAsync(BeforeBatchSaveArgs<EditBuildingRiskVM> args)
        {
            try
            {
                //find the changes in the datagrid and then apply them to underlying datamodel object -buildingRiskVM
                if (args.BatchChanges.ChangedRecords.Count != 0)
                {
                    foreach (var rec in args.BatchChanges.ChangedRecords)
                    {
                        var obj = buildingRiskVm.Find(x => x.BuildingNumber == rec.BuildingNumber);
                        if (obj != null)
                        {
                            obj.ThisMonthRiskComment = rec.ThisMonthRiskComment;
                            obj.ThisMonthRiskScore = rec.ThisMonthRiskScore;
                        }
                    }
                }

                // now write whole datamodel into the db as we want everything saved for audit purposes
                string strResult = await UpdateBuildingRiskData();
                if (strResult.Contains("Error") || strResult.Contains("denied"))
                {
                    showingDbError = true;
                }
                else
                {
                    showingDbSuccess = true;
                }
            }
            catch (Exception e)
            {
                showingDbError = true;
                Console.WriteLine(e.Message);
                throw;
            }
        }

        /// <summary>
        /// Awaitable task that writes whole data model to the database
        /// </summary>
        /// <returns></returns>
        public async Task<string> UpdateBuildingRiskData()
        {
            string strResult = "";
            if (buildingRiskVm != null)
            {
                int monthValue = DateTime.Today.Month;
                string strAudit = DateTime.Now.ToString() + " " + selectedManager.ToString();

                //cycle through each item in the datamodel and write updates back to the database
                foreach (EditBuildingRiskVM building in buildingRiskVm)
                {
                    if (building.RecordNumber != 0)
                    {
                        strResult = await _buildingDb.UpdateBuildingMonthRisk(building.RecordNumber, building.ThisMonthRiskScore.Value, monthValue, strAudit, building.ThisMonthRiskComment);
                    }
                    if (strResult.Contains("Error") || strResult.Contains("denied"))
                    {
                        break;
                    }
                }
            }
            else
            {
                strResult = "Error";
            }
            return strResult;
        }

       #endregion //Data manipulation

        #region Grid look and feel
        public void RowBound(RowDataBoundEventArgs<EditBuildingRiskVM> args)
        {
            if (args.Data.ThisMonthRiskScore == 0)
            {
                //args.Row.AddClass(new string[] { "below-25" });
            }
            else if (args.Data.ThisMonthRiskScore < 6)
            {
                args.Row.AddClass(new string[] { "below-6" });
            }
            else if (args.Data.ThisMonthRiskScore < 9)
            {
                args.Row.AddClass(new string[] { "below-9" });
            }
            else if (args.Data.ThisMonthRiskScore <= 10)
            {
                args.Row.AddClass(new string[] { "above-9" });
            }
            else
            {
                //do nothing - apply default style
            }
        }

        /// <summary>
        /// Contains settings for the textbox editor on for risk comments
        /// </summary>
        public IEditorSettings RiskCommentEditor = new StringEditCellParams{
            Params = new TextBoxModel() { ShowClearButton = false, Multiline = true }
        };
        #endregion
    }
}
